#!/bin/bash
#
#  File: sshirw.sh
#
#  Copyright 2024-2025 Penguin Computing Inc.  All Rights Reserved.
#
# This script creates an ssh tunnel to a remote irw server 
# and launches X, gnome-session and irw on the irw server 
# and opens a browser session with the remote irw server
#
# Just make sure the requested X-server is properly installed and configured
# on the remote system

set +e

# Default values
# IP Address or host name to reach the server from the internet:
EXTERNAL_SERVER_IP=NOT-SET

# Port number ssh daemon is listening:
EXTERNAL_SERVER_SSH_PORT=22

# User name on server:
SERVER_USERNAME=NOT-SET

# IP Address or host name of the server within the VPN:
INTERNAL_SERVER_IP=127.0.0.1

# Port ICE RemoteWare server runs on
SERVER_IRW_PORT=8890

# Localhost address the client/browser can connect to server
CLIENT_IP=127.0.0.1

WM_TITLE="Xephyr ${USER}"

# Defaults:
HOST_DISPLAY_ID=:1
GUEST_DISPLAY_ID=:16

CLIENT_IRW_PORT=8891

WM_TITLE="Xephyr ${USER}"
XSERVER="Xvfb"

START_GNOME_SHELL=0

GNOME_SHELL_OPTIONS=--x11
GNOME_SESSION_OPTIONS=

START_CLIENT=1

VERBOSE=1

MINIMIZE_WINDOW=NOT-SET

function vecho() {
    if [[ ${VERBOSE} -ge $1 ]]; then
        echo "$2"
    fi
}

function verror() {
    bg_red=`tput setab 1`
    white=`tput setaf 15`
    reset=`tput sgr0`
    echo
    echo ${bg_red}${white}"ERROR: $1"${reset}
    exit_script
}

function help() {
    echo
    echo "Usage: ssh-irw.sh [-h] -se <ip address> -u <user name>  [-s <port number>]  [-si <ip address>]  [-ci <ip address>]  [-p <port number>]  [-sp <port number>] [-x <executable>]  [-G <display id>]  [-H <display id>]  [-v <level>]  [-d]  [-gs]  [-o <options>]  [-n]"
    echo
    echo "sshirw.sh connects a ssh-vpn tunnel between server and client, launches a nested guest"
    echo "x server session and starts a private irw session without any authentification."
    echo
    echo "  Options"
    echo "   -h, --help                     this help"
    echo "   -se  <ip address>              external server ip address (globally accessible)"
    echo "   -u, --user <user name>         server site user name"
    echo "   -si  <ip address>              internal server ip address (vpn internal)"
    echo "   -ci, --client-ip  <ip address>              client ip address"
    echo "   -x, --xserver <executable>     window manager in /usr/bin, default 'Xvfb'"
    echo "   -s, --ssh-port <port number>   port number of ssh daemon on server"
    echo "   -sp, --server-irw-port <port>  irw port on server (internal only)"
    echo "   -p, --client-irw-port <port>   irw port on client"
    echo "   -G, --guestid <display id>     display id of guest x11 session, e.g. ':16'"
    echo "   -H, --hostid <display id>      display id of x11 session on host"
    echo "   -gs, --gnome-session           use 'gnome-session' as window manager (default 'gnome-shell')"
    echo "   -o, --opts <options>           options to window manager, can be empty to clear default options"
    echo "   -v, --verbose <level>          sets verbosity"
    echo "   -d, --debug                    debug mode"
    echo
    echo "i.e. ssh-irw.sh -se 10.1.0.22  -u userA  -x Xephyr  -p 8423  -sp 8443  -H :1  -G :12"
    echo "     ssh-irw.sh -se 10.1.0.22  -u userB  -x Xvfb  -p 8423  -sp 8443  -H :1  -G :12  -v 2"
    echo
}

print_params() {
    echo
    echo "Parameters:"
    echo "EXTERNAL_SERVER_IP, -se      = ${EXTERNAL_SERVER_IP}"
    echo "USERNAME, -u                 = ${SERVER_USERNAME}"
    echo "EXTERNAL_SERVER_SSH_PORT, -s = ${EXTERNAL_SERVER_SSH_PORT}"
    echo "INTERNAL_SERVER_IP, -si      = ${INTERNAL_SERVER_IP}"
    echo "XSERVER, -x                  = ${XSERVER}"
    echo "MINIMIZE_WINDOW, -m          = ${MINIMIZE_WINDOW}"
    echo "SERVER_IRW_PORT, -sp         = ${SERVER_IRW_PORT}"
    echo "CLIENT_IP                    = ${CLIENT_IP}"
    echo "CLIENT_IRW_PORT, -p          = ${CLIENT_IRW_PORT}"
    echo "GUEST_DISPLAY_ID, -G         = ${GUEST_DISPLAY_ID}"
    echo "HOST_DISPLAY_ID, -H          = ${HOST_DISPLAY_ID}"
    echo "CLIENT_IRW_PORT, -p          = ${CLIENT_IRW_PORT}"
    echo "START_GNOME_SHELL, -gs       = ${START_GNOME_SHELL}"
    echo "GNOME_SESSION_OPTIONS, -o    = ${GNOME_SESSION_OPTIONS}"
    echo "GNOME_SHELL_OPTIONS, -o      = ${GNOME_SHELL_OPTIONS}"
    echo "START_CLIENT, -n             = ${START_CLIENT}"
    echo "VERBOSE, -v                  = ${VERBOSE}"
    echo "DEBUG, -d                    = ${DEBUG}"
}

check_params() {
    if [[ "${EXTERNAL_SERVER_IP}" == "NOT-SET" ]]; then
        verror "Please use '-se' option or set 'EXTERNAL_SERVER_IP' variable to set external ip/host address of server";
    fi
    if [[ "${SERVER_USERNAME}" == "NOT-SET" ]]; then
        verror "Please use '-u, --user-name' option or set SERVER_USERNAME variable to user account name on server"
    fi
    if [[ ${MINIMIZE_WINDOW_SET} -eq 0 ]]; then
        if [[ ${XSERVER} == "Xephyr" ]]; then
            MINIMIZE_WINDOW=1
        else
            MINIMIZE_WINDOW=0            
        fi
    fi
    if [[ ${VERBOSE} -gt 1 ]]; then
        if [[ "${EXTERNAL_SERVER_SSH_PORT}" == "22" ]]; then
            echo
            echo "Using default ssh port '${EXTERNAL_SERVER_SSH_PORT}' to connect to sshd on server"
            echo "  Please check this value, if any connectivity problems occure"
        fi
        if [[ ${SERVER_IRW_PORT} == "8890" ]]; then
            echo
            echo "Using default port ${SERVER_IRW_PORT}' to connect to irw server"
            echo "  Please modify this value for each client connecting to the server"
        fi
        if [[ ${CLIENT_IRW_PORT} == "8891" ]]; then
            echo
            echo "Using default port '${CLIENT_IRW_PORT}' on the client to connect to irw server"
            echo "  Please use for each server a different value when you connect from this client"
        fi
        if [[ ${HOST_DISPLAY_ID} == ":1" ]]; then
            echo
            echo "Using default host display id ${HOST_DISPLAY_ID}' to start X server"
            echo "  Please check, if this value is valid for the X session on the server"
        fi
        if [[ ${GUEST_DISPLAY_ID} == ":16" ]]; then
            echo
            echo "Using default guest display id ${GUEST_DISPLAY_ID}' to start X server"
            echo "  Please modify this value for each client connecting to the server"
        fi
    fi
}

trap exit_script INT
trap exit_script TERM

# Prevent recursive calls of exit_script 
exit_called=0

# Local processes:
ssh_pid=0

# Remote processes:
x11_pid=0
server_pid=0

function exit_script() {

    if [ ${exit_called} -eq 1 ]; then
        exit 0
    else
        exit_called=1
    fi

    echo "Exiting"

    # Terminate X server using stored pid
    if [[ ${x11_pid} -gt 0 ]] ; then
        ssh -p ${EXTERNAL_SERVER_SSH_PORT} ${SERVER_USERNAME}@${EXTERNAL_SERVER_IP} kill -s SIGTERM ${x11_pid}
    fi

    # Terminate irw
    if [[ ${server_pid} -gt 0 ]] ; then
        ssh -p ${EXTERNAL_SERVER_SSH_PORT} ${SERVER_USERNAME}@${EXTERNAL_SERVER_IP} kill -s SIGTERM ${server_pid}
    fi

    # Terminate ssh tunnel and gnome-session
    if [[ ${ssh_pid} -gt 0 ]] ; then
        kill -s SIGTERM ${ssh_pid}
    fi

    exit 0
}

while [[ $# -gt 0 ]]
do
    key="$1"

    case $key in
        -h|--help)
            help
            exit 0
            ;;
        -se)
            EXTERNAL_SERVER_IP="$2"
            shift # past argument
            ;;
        -si)
            INTERNAL_SERVER_IP="$2"
            shift
            ;;
        -ci|--client-ip)
            CLIENT_IP="$2"
            shift
            ;;
        -sp|--server-irw-port)
            SERVER_IRW_PORT="$2"
            shift
            ;;
        -p|--client-irw-port)
            CLIENT_IRW_PORT="$2"
            shift
            ;;
        -u|--user)
            SERVER_USERNAME="$2"
            shift
            ;;
        -s|--ssh-port)
            EXTERNAL_SERVER_SSH_PORT="$2"
            shift
            ;;
        -x|--xserver)
            XSERVER="$2"
            shift # past argument
            ;;
        -G|--guestid)
            GUEST_DISPLAY_ID="$2"
            shift
            ;;
        -H|--hostid)
            HOST_DISPLAY_ID="$2"
            shift
            ;;
        -v|--verbose)
            VERBOSE="$2"
            shift
            ;;
        -gs|--gnome-session)
            START_GNOME_SHELL=0;
            ;;
        -o|--opts)
            GNOME_SHELL_OPTIONS="$2";
            GNOME_SESSION_OPTIONS="$2";
            shift
            ;;
        -n|--no-client)
            START_CLIENT=0
            ;;
        -m|--minimize-window)
            MINIMIZE_WINDOW_SET=1
            MINIMIZE_WINDOW=1
            ;;
        *)
            help
            verror "unkown option: $key" 
    esac
    shift # past arg or value
done

check_params

if [[ ${VERBOSE} -ge 1 ]]; then
    print_params
fi


if [[ ${DEBUG} -ge 1 ]]; then
    vecho 4 "Testing remote connecitivity and querying environment"
    vecho 4 " => ssh -t -t -p ${EXTERNAL_SERVER_SSH_PORT} ${SERVER_USERNAME}@${EXTERNAL_SERVER_IP} bash -c env"
    vecho 4
    ssh -t -t -p ${EXTERNAL_SERVER_SSH_PORT} ${SERVER_USERNAME}@${EXTERNAL_SERVER_IP} bash -c env

    if [[ $? -ne 0 ]]; then
        verror "Can't connect to ${EXTERNAL_SERVER_IP} as user ${SERVER_USERNAME} at port ${EXTERNAL_SERVER_SSH_PORT}"
    fi
    vecho 4
fi

if [[ ${MINIMIZE_WINDOW} -eq 1 ]]; then
    if [[ ! -f /usr/bin/xdotool ]]; then
        vecho 1 "/usr/bin/xdotool not found, will not try to minimize the guest window"
        MINIMIZE_WINDOW=0
    fi
fi

# Launch X, gnome-session, irw on ssh server
vecho 1 
vecho 1 "Launch X-server (${XSERVER}) ..."

XSERVER_OPTIONS=${GUEST_DISPLAY_ID}

if [[ ${XSERVER} == "Xephyr" ]]; then
    XSERVER_OPTIONS="-br -ac -noreset -title "${WM_TITLE}" -screen 1920x1080 ${GUEST_DISPLAY_ID} -dpi 96"
fi

vecho 4
vecho 4 "Launch command: /usr/bin/${XSERVER} ${XSERVER_OPTIONS}"
vecho 4

# launch X, gnome-session, irw on ssh server and remember pid of X
# X does not terminate at SIGHUP, which is why X is launched 
# separately from the other processes
eval "$(ssh -t -t -p ${EXTERNAL_SERVER_SSH_PORT} ${SERVER_USERNAME}@${EXTERNAL_SERVER_IP} "$(cat <<EOF
DISPLAY=${HOST_DISPLAY_ID} /usr/bin/${XSERVER} ${XSERVER_OPTIONS} &> /dev/null &
x11_pid=\$!
sleep 3
declare -p x11_pid
EOF
)")"

# remove newline at end of string
x11_pid=${x11_pid::${#x11_pid}-1}

if [[ ${x11_pid} -gt 0 ]]; then
    vecho 1 "... started x-server (pid ${x11_pid})"
else
    verror "... failed to start x-server"
fi

sleep 1

if [[ ${MINIMIZE_WINDOW} -eq 1 ]]; then
    vecho 1 "Trying to minimize x server window"
    ssh -t -t -p ${EXTERNAL_SERVER_SSH_PORT} ${SERVER_USERNAME}@${EXTERNAL_SERVER_IP} \
        DISPLAY=${HOST_DISPLAY_ID} /usr/bin/xdotool search "${WM_TITLE}" windowminimize
    vecho 1 " ... minimized guest window"
fi

sleep 2

# When using gnome-session, to avoid auth dialog on server (rocky 9), see
# https://www.reddit.com/r/Ubuntu/comments/15stmwn/how_do_i_suppress_authentication_is_required_to/?rdt=42444

if [[ ${START_GNOME_SHELL} -eq 1 ]]; then
    WM_NAME=gnome-shell
    WM_OPTIONS=${GNOME_SHELL_OPTIONS}
else
    WM_NAME=gnome-session
    WM_OPTIONS=${GNOME_SESSION_OPTIONS}
fi

vecho 1
vecho 1 "Launch ${WM_NAME} and ICE RemoteWare for display '${GUEST_DISPLAY_ID}' ..."

WM_LOGFILE="/dev/null"

vecho 2
vecho 2 "ice-remoteware options: --application  --setConfig=Server.X11.Display=${GUEST_DISPLAY_ID} --setConfig=Server.Network.Port=${CLIENT_IRW_PORT}"
vecho 2 "      --setConfig=Server.Network.Secure=false  --setConfig=Server.Log.FileName=~/ice-remoteware.log --setConfig=Server.Misc.LocalHostRestricted=false  --setConfig=Server.Auth.Enabled=false"

ssh -p ${EXTERNAL_SERVER_SSH_PORT} -L ${CLIENT_IP}:${CLIENT_IRW_PORT}:${INTERNAL_SERVER_IP}:${SERVER_IRW_PORT} ${SERVER_USERNAME}@${EXTERNAL_SERVER_IP} \
"DISPLAY=${GUEST_DISPLAY_ID} dbus-run-session ${WM_NAME} ${WM_OPTIONS} > ${WM_LOGFILE} 2>&1  &
sleep 1

DISPLAY=${GUEST_DISPLAY_ID} /opt/ice-remoteware/bin/ice-remoteware \
    --application \
    --setConfig=Server.X11.Display=${GUEST_DISPLAY_ID} \
    --setConfig=Server.Network.Port=${SERVER_IRW_PORT} \
    --setConfig=Server.Network.Secure=false \
    --setConfig=Server.Misc.LocalHostRestricted=false \
    --setConfig=Server.Log.FileName=~/ice-remoteware-"${GUEST_DISPLAY_ID:1}".log \
    --setConfig=Server.Auth.Enabled=false &> /dev/null" &

ssh_pid=$!
sleep 6

vecho 1 "... ${WM_NAME} and ICE RemoteWare launched"
vecho 1 ""

opened=0

vecho 0 "Server accessible at http://${CLIENT_IP}:${CLIENT_IRW_PORT}"

if [[ ${START_CLIENT} -eq 1 ]]; then
    if [[ $(uname) == "Darwin" ]]; then
        if [[ -d  "/Applications/ice-remoteware-client.app/" ]]; then
            open irw://${CLIENT_IP}:${CLIENT_IRW_PORT}?secure=false
            if [[ $? -eq 0 ]]; then
                opened=1
            fi
        fi
        if [[ ${opened} -eq 0 ]]; then
            open http://${CLIENT_IP}:${CLIENT_IRW_PORT}
        fi
    else
        uname | grep -is CYGWIN > /dev/null
        if [[ $? -eq 0 ]]; then
            if [[ -d  "${PROGRAMFILES//\\//}/Penguin Solutions/ICE RemoteWare Client" ]]; then
                cmd /c start "" irw://${CLIENT_IP}:${CLIENT_IRW_PORT}?secure=false
                if [[ $? -eq 0 ]]; then
                    opened=1
                fi
            fi
            if [[ ${opened} -eq 0 ]]; then
                cmd /c start "" http://${CLIENT_IP}:${CLIENT_IRW_PORT}
            fi
        else
            if [[ -d  "/opt/ice-remoteware-client/" ]]; then
                xdg-open irw://${CLIENT_IP}:${CLIENT_IRW_PORT}?secure=false > /dev/null 2>&1
                if [[ $? -eq 0 ]]; then
                    opened=1
                fi
            fi
            if [[ ${opened} -eq 0 ]]; then
                xdg-open http://${CLIENT_IP}:${CLIENT_IRW_PORT} > /dev/null 2>&1
            fi
        fi
    fi
else
    vecho 2 "Not starting connecting to server"
fi

vecho 0 "press ENTER or Ctrl-C to exit"
read
exit_script
